//
//  TodoListEditor.swift
//  Do It
//
//  Created by Jim Dovey on 2/4/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

struct TodoListEditor: View {
    @Environment(\.presentationMode) private var presentation
    @EnvironmentObject private var data: DataCenter
    @State var list: TodoItemList

    var iconGradient: LinearGradient {
        var (h, s, b) = list.color.hsb
        b *= 0.7
        
        return LinearGradient(
            gradient: Gradient(colors: [
                list.color.uiColor,
                Color(hue: h, saturation: s, brightness: b)
            ]),
            startPoint: .topLeading,
            endPoint: .bottom)
    }

    var body: some View {
        VStack {
            HStack(alignment: .firstTextBaseline) {
                Button("Cancel") {
                    self.presentation.wrappedValue.dismiss()
                }
                Spacer()
                Text("Name & Appearance")
                    .bold()
                Spacer()
                Button(action: {
                    self.data.updateList(self.list)
                    self.presentation.wrappedValue.dismiss()
                }) {
                    Text("Done")
                        .bold()
                }
            }
            .padding()

            Image(systemName: list.icon)
                .font(.system(size: 56, weight: .bold, design: .rounded))
                .aspectRatio(contentMode: .fit)
                .padding(36)
                .foregroundColor(.white)
                .frame(width: 112, height: 112)
                .background(iconGradient)
                .clipShape(Circle())
                .modifier(DoubleShadow())
                .accessibility(hidden: true)    // Image(decorative:) equivalent

            TextField("List Title", text: $list.name)
                .font(.system(size: 20, weight: .semibold, design: .rounded))
                .modifier(BorderedTextField())
                .padding()
            
            VStack(spacing: 0) {
                Divider()
                ScrollView {
                    ColorPicker(selectedColor: $list.color)
                        .padding()
                    IconChooser(selectedIcon: $list.icon)
                        .padding()
                }
            }
        }
    }
}

struct TodoListEditor_Previews: PreviewProvider {
    static var previews: some View {
        TodoListEditor(list: defaultTodoLists[3])
            .environmentObject(DataCenter())
    }
}
